#!/usr/bin/env python

import sys
from math import pi, log10
from gnuradio.eng_notation import num_to_str

"""
The input power Pr received by an antenna is either:
Pr = PD * Ae                        (if effective antenna area Ae is known)
Pr = PD * G * wavelength^2 /(4*pi)  (if antenna gain G is known)
Where Pr= input power received
PD is power density in Watts/m^2
Ae is effective antenna area in m^2
G is antenna Gain (dimensionless)

if we assume G = 1.0  (isotropic radiator?)
then 
Pr = PD *  wavelength * wavelength /(4*pi)

or the other way around
PD = Pr * 4 * pi/(wavelength * wavelength)
"""
def power_density_to_power(power_density, wavelength):
    """
    Calculate received power in Watts
    Assume isotropic radiator (spherical radiation pattern, gain is 1.0).

    @param power_density: Watts/meter^2
    @param wavelength: wavelength of the signal = speed_of_light/frequency where speed_of_light is in meters/second = 1.0e8

    """
    return power_density *  wavelength * wavelength /(4.0*pi)

def power_to_power_density(power, wavelength):
    """
    Calculate received power_density in Watts/m^2
    Assume isotropic radiator (spherical radiation pattern, gain is 1.0).

    @param power_density: Watts/meter
    @param wavelength: wavelength of the signal = speed_of_light/frequency where speed_of_light is in meters/second = 1.0e8

    """
    return power * 4.0 * pi/(wavelength * wavelength)

def power_density(transmitted_power, distance):
    """
    Estimate power density in Watts/meter^2.
    Assume isotropic radiator (spherical radiation pattern).

    @param transmitted_power: Watts
    @param distance: distance from transmitter in meters

    """
    return transmitted_power / (4 * pi * distance * distance)

def power_density_free_space(transmitted_power, distance):
    """
    Estimate power density in Watts/meter^2.
    Assume isotropic radiator (spherical radiation pattern).

    @param transmitted_power: Watts
    @param distance: distance from transmitter in meters

    """
    wavelength=1.0 #will cancel out in the two calculations so can be set to unity
    return power_to_power_density(transmitted_power / path_loss_free_space_model(distance,wavelength), wavelength)

def power_density_flat_earth(transmitted_power, distance, h1, h2, wavelength):
    """
    Estimate power density in Watts/meter^2.
    Assume isotropic radiator (spherical radiation pattern).

    @param transmitted_power: Watts
    @param distance: distance from transmitter in meters

    """
    return power_to_power_density(transmitted_power / path_loss_flat_earth_and_free_space_model(distance,h1,h2, wavelength),wavelength)


def path_loss_flat_earth_mode(distance,h1,h2):
    """
    Estimate power loss with flat-earth model (dimensionless).
    Assume isotropic radiator (spherical radiation pattern).
    assume distance > dbrk
    where dbrk is break distance
    dbrk= 4 * pi * h1 * h2/wavelength

The free space formula also applies for propagation
along a flat earth between antennas at heights h1, h2, provided that d is less than a
breakpoint distance dbrk= 4 * pi * h1 * h2/wavelength. However, when d > dbrk, the
ground-reflected ray is almost the same length as the direct ray. Since the reflection
coefficient is close to -1, the two rays almost cancel. In this case, we find that the
path loss increases as the fourth power of the distance

    @param distance: distance between transmitter and receiver in meters
    @param h1: height of receiving antenna in meters
    @param h2: height of transmitting antenna in meters
    """
    pathloss=distance*distance*distance*distance/(h1*h2)
    return pathloss

def path_loss_free_space_model(distance,wavelength):
    """
    Estimate power loss in free space (dimensionless).
    Assume isotropic radiator (spherical radiation pattern).

    @param distance: distance between transmitter and receiver in meters
    @param wavelength: wavelength of the signal = speed_of_light/frequency where speed_of_light is in meters/second = 1.0e8
    """
    tmp = 4*pi*distance/wavelength
    pathloss=tmp*tmp
    return pathloss

def path_loss_flat_earth_and_free_space_model(distance,h1,h2, wavelength):
    """
    Estimate power loss with flat-earth model (dimensionless).
    Assume isotropic radiator (spherical radiation pattern).
    dbrk is break distance
    dbrk= 4 * pi * h1 * h2/wavelength

Both freespace and flat_earth path loss formulas give the same
result when d = dbrk. The free space and flat earth path loss formulas can be combined
into one formula

    @param distance: distance between transmitter and receiver in meters
    @param h1: height of receiving antenna in meters
    @param h2: height of transmitting antenna in meters
    @param wavelength: wavelength of the signal = speed_of_light/frequency where speed_of_light is in meters/second = 1.0e8
    """
    pathloss_free_space=path_loss_free_space_model(distance,wavelength)

    dbrk= 4.0 * pi * h1 * h2/wavelength
    #pathloss_flat_earth=distance*distance*distance*distance/(h1*h2)
    tmp2=1.0+distance/dbrk
    flat_earth_correction=tmp2*tmp2

    pathloss=pathloss_free_space*flat_earth_correction

    return pathloss

def ratio_of_target_return_to_direct(Rl, Rt, Rr, rcs):
    """
    Estimate relative strength of signal levels for direct and reflected
    path in bistatic radar.  Assume all antenna factors are constant,
    and hence 'wash out'.  Also assume that the antennas are isotropic
    radiators (spherical radiation pattern).

    @param Rl:  distance between Tx and Rx in meters
    @param Rt:  distance between Tx and target in meters
    @param Rr:  distance between Rx and target in meters
    @param rcs: radar cross section in meters^2

    @returns: ratio of reflected to direct in decibels
    """
    Tx_power = 1
    direct_path_power_density = power_density(Tx_power, Rl)
    power_at_target = power_density(Tx_power, Rt) * rcs
    reflected_path_power_density = power_density(power_at_target, Rr)
    return 10*log10(reflected_path_power_density / direct_path_power_density)

def ratio_of_target_return_to_direct_free_space(Rl, Rt, Rr, rcs):
    """
    Estimate relative strength of signal levels for direct and reflected
    path in bistatic radar.  Assume all antenna factors are constant,
    and hence 'wash out'.  Also assume that the antennas are isotropic
    radiators (spherical radiation pattern).

    @param Rl:  distance between Tx and Rx in meters
    @param Rt:  distance between Tx and target in meters
    @param Rr:  distance between Rx and target in meters
    @param rcs: radar cross section in meters^2

    @returns: ratio of reflected to direct in decibels
    """
    Tx_power = 1
    direct_path_power_density = power_density_free_space(Tx_power, Rl)
    power_at_target = power_density_free_space(Tx_power, Rt) * rcs
    reflected_path_power_density = power_density_free_space(power_at_target, Rr)
    return 10*log10(reflected_path_power_density / direct_path_power_density)

def ratio_of_target_return_to_direct_only_flat_earth(Rl, Rt, Rr, rcs,hrec,htrans,hplane,wavelength):
    """
    Estimate relative strength of signal levels for direct and reflected
    path in bistatic radar.  Assume all antenna factors are constant,
    and hence 'wash out'.  Also assume that the antennas are isotropic
    radiators (spherical radiation pattern).

    @param Rl:  distance between Tx and Rx in meters
    @param Rt:  distance between Tx and target in meters
    @param Rr:  distance between Rx and target in meters
    @param rcs: radar cross section in meters^2

    @returns: ratio of reflected to direct in decibels
    """
    Tx_power = 1
    direct_path_power_density = power_density_flat_earth(Tx_power, Rl,hrec, htrans, wavelength)
    power_at_target = power_density_flat_earth(Tx_power, Rt, htrans, hplane, wavelength) * rcs
    reflected_path_power_density = power_density_flat_earth(power_at_target, Rr, hplane, hrec, wavelength)
    return 10*log10(reflected_path_power_density / direct_path_power_density)


def ratio_of_target_return_to_direct_flat_earth(Rl, Rt, Rr, rcs,hrec,htrans,hplane,wavelength):
    """
    Estimate relative strength of signal levels for direct and reflected
    path in bistatic radar.  Assume all antenna factors are constant,
    and hence 'wash out'.  Also assume that the antennas are isotropic
    radiators (spherical radiation pattern).

    @param Rl:  distance between Tx and Rx in meters
    @param Rt:  distance between Tx and target in meters
    @param Rr:  distance between Rx and target in meters
    @param rcs: radar cross section in meters^2

    @returns: ratio of reflected to direct in decibels
    """
    Tx_power = 1
    direct_path_power_density = power_density_flat_earth(Tx_power, Rl,hrec, htrans, wavelength)
    power_at_target = power_density_free_space(Tx_power, Rt) * rcs
    reflected_path_power_density = power_density_free_space(power_at_target, Rr)
    return 10*log10(reflected_path_power_density / direct_path_power_density)

def print_header(f):
    f.write("Rl:  distance between Tx and Rx in meters\n")
    f.write("Rt:  distance between Tx and target in meters\n")
    f.write("Rr:  distance between Rx and target in meters\n")
    f.write("rcs: radar cross section in meters^2\n")
    f.write("   Rl      Rt      Rr      rcs      dB\n")
    f.write("-----------------------------------------\n")
    
def print_result(f, Rl, Rt, Rr, rcs, dB):
    f.write("%6sm %6sm %6sm %6s  %6s\n" %  tuple([num_to_str(x) for x in [Rl, Rt, Rr, rcs, dB]]))

def calc_print(f, Rl, Rt, Rr, rcs):
    print_result(f, Rl, Rt, Rr, rcs, ratio_of_target_return_to_direct(Rl, Rt, Rr, rcs))

def calc_print_free_space(f, Rl, Rt, Rr, rcs):
    print_result(f, Rl, Rt, Rr, rcs, ratio_of_target_return_to_direct_free_space(Rl, Rt, Rr, rcs))

def calc_print_earth(f, Rl, Rt, Rr, rcs,hrec,htrans,hplane,wavelength):
    print_result(f, Rl, Rt, Rr, rcs, ratio_of_target_return_to_direct_flat_earth(Rl, Rt, Rr, rcs,hrec,htrans,hplane,wavelength))
    
def main():
    f = sys.stdout
    print_header(f)
    calc_print(f, 40e3,1000e3,1000e3, 10.0)
    calc_print(f, 40e3, 300e3, 300e3, 10.0)
    calc_print(f, 40e3, 100e3, 100e3, 10.0)
    calc_print(f, 40e3,  80e3,  80e3, 10.0)
    calc_print(f, 40e3,  40e3,  40e3, 10.0)
    calc_print(f, 40e3,  40e3,   8e3, 10.0)
    calc_print(f, 40e3,  60e3,  20e3, 10.0)
    calc_print(f, 40e3,  20e3,  60e3, 10.0)
    calc_print(f, 40e3,  20e3,  20e3, 10.0)

    calc_print_free_space(f, 40e3,1000e3,1000e3, 10.0)
    calc_print_free_space(f, 40e3, 300e3, 300e3, 10.0)
    calc_print_free_space(f, 40e3, 100e3, 100e3, 10.0)
    calc_print_free_space(f, 40e3,  80e3,  80e3, 10.0)
    calc_print_free_space(f, 40e3,  40e3,  40e3, 10.0)
    calc_print_free_space(f, 40e3,  40e3,   8e3, 10.0)
    calc_print_free_space(f, 40e3,  60e3,  20e3, 10.0)
    calc_print_free_space(f, 40e3,  20e3,  60e3, 10.0)
    calc_print_free_space(f, 40e3,  20e3,  20e3, 10.0)

    freq=100.0e6 # 100 MHz
    hrec=10.0
    htrans=300.0
    hplane=10000.0
    wavelength=3.0e8/freq # is 3 meter for FM stations
    calc_print_earth(f, 40e3,1000e3,1000e3, 10.0,hrec,htrans,hplane,wavelength)
    calc_print_earth(f, 40e3, 300e3, 300e3, 10.0,hrec,htrans,hplane,wavelength)
    calc_print_earth(f, 40e3, 100e3, 100e3, 10.0,hrec,htrans,hplane,wavelength)
    calc_print_earth(f, 40e3,  80e3,  80e3, 10.0,hrec,htrans,hplane,wavelength)
    calc_print_earth(f, 40e3,  40e3,  40e3, 10.0,hrec,htrans,hplane,wavelength)
    calc_print_earth(f, 40e3,  40e3,   8e3, 10.0,hrec,htrans,hplane,wavelength)
    calc_print_earth(f, 40e3,  60e3,  20e3, 10.0,hrec,htrans,hplane,wavelength)
    calc_print_earth(f, 40e3,  20e3,  60e3, 10.0,hrec,htrans,hplane,wavelength)
    calc_print_earth(f, 40e3,  20e3,  20e3, 10.0,hrec,htrans,hplane,wavelength)

    
if __name__ == '__main__':
    main()
